/* Akemi Kitchen Prep - Frontend JavaScript
   Akemi Tech Labs */

// -- State --
var pendingFiles = [];
var processedOrders = [];
var prepData = {};
var orderDate = '';

// -- Helpers --
function esc(s) {
    return String(s)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;');
}

function toast(msg) {
    var t = document.getElementById('toast');
    if (!t) return;
    t.textContent = msg;
    t.classList.add('show');
    setTimeout(function() { t.classList.remove('show'); }, 2500);
}

function showBanner(type, msg) {
    var b = document.getElementById('statusBanner');
    if (!b) return;
    b.className = 'status-banner ' + type;
    b.style.display = 'flex';
    var icon = document.getElementById('statusIcon');
    if (type === 'processing') {
        icon.innerHTML = '<span class="spinner"></span>';
    } else if (type === 'success') {
        icon.textContent = 'OK';
    } else if (type === 'error') {
        icon.textContent = '!';
    } else {
        icon.textContent = '';
    }
    document.getElementById('statusText').textContent = msg;
}

function hideBanner() {
    var b = document.getElementById('statusBanner');
    if (b) b.style.display = 'none';
}

// -- Dropzone --
(function initDropzone() {
    var dz = document.getElementById('dropzone');
    var fi = document.getElementById('fileInput');
    if (!dz || !fi) return;

    dz.addEventListener('click', function() { fi.click(); });
    dz.addEventListener('dragover', function(e) {
        e.preventDefault();
        dz.classList.add('dragover');
    });
    dz.addEventListener('dragleave', function() {
        dz.classList.remove('dragover');
    });
    dz.addEventListener('drop', function(e) {
        e.preventDefault();
        dz.classList.remove('dragover');
        addFiles(e.dataTransfer.files);
    });
    fi.addEventListener('change', function() {
        addFiles(fi.files);
        fi.value = '';
    });
})();

function addFiles(fileList) {
    for (var i = 0; i < fileList.length; i++) {
        var f = fileList[i];
        if (f.name.toLowerCase().endsWith('.pdf')) {
            pendingFiles.push(f);
        }
    }
    renderFileChips();
    var actions = document.getElementById('uploadActions');
    if (actions) actions.style.display = pendingFiles.length ? 'flex' : 'none';
}

function renderFileChips() {
    var c = document.getElementById('fileChips');
    if (!c) return;
    var html = '';
    for (var i = 0; i < pendingFiles.length; i++) {
        var f = pendingFiles[i];
        var kb = Math.round(f.size / 1024);
        html += '<div class="file-chip">' +
            '<span class="name">' + esc(f.name) + '</span>' +
            '<span class="size">' + kb + ' KB</span>' +
            '<button class="remove" onclick="removeFile(' + i + ')">&times;</button>' +
            '</div>';
    }
    c.innerHTML = html;
}

function removeFile(idx) {
    pendingFiles.splice(idx, 1);
    renderFileChips();
    var actions = document.getElementById('uploadActions');
    if (actions) actions.style.display = pendingFiles.length ? 'flex' : 'none';
}

function clearAll() {
    pendingFiles = [];
    processedOrders = [];
    prepData = {};
    orderDate = '';
    renderFileChips();
    hideBanner();
    var actions = document.getElementById('uploadActions');
    if (actions) actions.style.display = 'none';
    var summary = document.getElementById('orderSummary');
    if (summary) summary.style.display = 'none';
    var zones = document.getElementById('zoneSelection');
    if (zones) zones.style.display = 'none';
    var preview = document.getElementById('prepPreview');
    if (preview) preview.style.display = 'none';
}

// -- Process Orders --
function processOrders() {
    if (!pendingFiles.length) return;

    var btn = document.getElementById('processBtn');
    if (btn) btn.disabled = true;
    showBanner('processing', 'Processing ' + pendingFiles.length + ' PDF file(s)...');

    var fd = new FormData();
    for (var i = 0; i < pendingFiles.length; i++) {
        fd.append('files', pendingFiles[i]);
    }

    fetch('/upload', { method: 'POST', body: fd })
        .then(function(r) { return r.json().then(function(d) { return {ok: r.ok, data: d}; }); })
        .then(function(res) {
            if (btn) btn.disabled = false;

            if (!res.ok) {
                var errMsg = res.data.error || 'Upload failed';
                showBanner('error', errMsg);
                if (res.data.warnings) {
                    res.data.warnings.forEach(function(w) { console.warn(w); });
                }
                return;
            }

            var d = res.data;
            processedOrders = d.orders || [];
            prepData = d.prep_data || {};
            orderDate = d.order_date || '';

            var msg = d.order_count + ' orders processed for ' + orderDate +
                      ' (' + d.total_guests + ' total guests)';
            showBanner('success', msg);
            toast(d.order_count + ' orders processed');

            if (d.warnings && d.warnings.length) {
                d.warnings.forEach(function(w) { console.warn('Warning:', w); });
            }

            renderOrderSummary();
            renderPrepPreview();

            var zones = document.getElementById('zoneSelection');
            if (zones) zones.style.display = 'block';
        })
        .catch(function(e) {
            if (btn) btn.disabled = false;
            showBanner('error', 'Network error: ' + e.message);
        });
}

function renderOrderSummary() {
    var container = document.getElementById('orderSummary');
    if (!container) return;
    container.style.display = 'block';

    var title = document.getElementById('summaryTitle');
    if (title) {
        title.textContent = processedOrders.length + ' Orders for ' + orderDate;
    }

    var cards = document.getElementById('orderCards');
    if (!cards) return;
    var html = '';
    for (var i = 0; i < processedOrders.length; i++) {
        var o = processedOrders[i];
        html += '<div class="order-card">' +
            '<span class="order-no">#' + esc(o.order_no) + '</span>' +
            '<span class="order-badge">' + o.guests + ' guests</span>' +
            '<span class="order-badge">' + esc(o.category) + '</span>' +
            '<span class="order-badge">' + esc(o.event_time) + '</span>' +
            '<span class="order-badge">' + o.item_count + ' items</span>' +
            '</div>';
    }
    cards.innerHTML = html;
}

// -- Prep Preview --
var ZONE_CSS = {
    'HOT LINE': 'hot',
    'COLD LINE': 'cold',
    'BEVERAGES': 'bev',
    'BAKERY/DESSERT': 'bakery'
};

var ZONE_ORDER = ['HOT LINE', 'COLD LINE', 'BAKERY/DESSERT', 'BEVERAGES'];

function renderPrepPreview() {
    var container = document.getElementById('prepContent');
    var wrapper = document.getElementById('prepPreview');
    if (!container || !wrapper) return;

    wrapper.style.display = 'block';
    var html = '';

    for (var zi = 0; zi < ZONE_ORDER.length; zi++) {
        var zone = ZONE_ORDER[zi];
        var zoneData = prepData[zone];
        if (!zoneData) continue;

        var cls = ZONE_CSS[zone] || '';
        html += '<div class="zone-section ' + cls + '">';
        html += '<div class="zone-header">' +
                '<span class="zone-dot zone-' + cls + '"></span>' +
                esc(zone) + '</div>';

        var categories = Object.keys(zoneData);
        for (var ci = 0; ci < categories.length; ci++) {
            var cat = categories[ci];
            var catData = zoneData[cat];
            if (!catData || !catData.items || !catData.items.length) continue;

            html += '<div class="cat-header">' + esc(cat.toUpperCase()) + '</div>';

            var orderRefs = [];
            for (var oi = 0; oi < catData.orders.length; oi++) {
                var o = catData.orders[oi];
                orderRefs.push(esc(o.order_no) + ' x' + o.guests);
            }
            html += '<div class="cat-orders">Orders: ' + orderRefs.join(', ') + '</div>';

            for (var ii = 0; ii < catData.items.length; ii++) {
                var item = catData.items[ii];
                var qtyParts = [];
                for (var qi = 0; qi < item.quantities.length; qi++) {
                    qtyParts.push(item.quantities[qi].qty);
                }
                html += '<div class="item-row">' +
                    '<span class="item-name">' + esc(item.name) + '</span>' +
                    '<span class="item-qty">' + qtyParts.join(', ') + '</span>' +
                    '<span class="item-total">= ' + item.total + '</span>' +
                    '</div>';
            }
        }
        html += '</div>';
    }

    if (!html) {
        html = '<div style="text-align:center;padding:40px;color:var(--text-muted)">' +
               '<p>No items found in uploaded orders.</p></div>';
    }
    container.innerHTML = html;
}

// -- Generate Report --
function generateReport() {
    var selected = [];
    var checks = document.querySelectorAll('.zone-check input:checked');
    for (var i = 0; i < checks.length; i++) {
        selected.push(checks[i].value);
    }
    if (!selected.length) {
        toast('Please select at least one zone');
        return;
    }

    showBanner('processing', 'Generating PDF report...');

    fetch('/generate', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ zones: selected })
    })
    .then(function(r) {
        if (!r.ok) {
            return r.json().then(function(d) { throw new Error(d.error || 'Generation failed'); });
        }
        var filename = 'Prep_List.pdf';
        var disp = r.headers.get('Content-Disposition');
        if (disp) {
            var match = disp.match(/filename[^;=\n]*=(['\"]?)([^'\";\n]*)\1/);
            if (match && match[2]) filename = match[2];
        }
        return r.blob().then(function(blob) { return {blob: blob, filename: filename}; });
    })
    .then(function(result) {
        hideBanner();
        toast('PDF generated: ' + result.filename);

        var url = URL.createObjectURL(result.blob);
        var a = document.createElement('a');
        a.href = url;
        a.download = result.filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        setTimeout(function() { URL.revokeObjectURL(url); }, 5000);
    })
    .catch(function(e) {
        showBanner('error', e.message);
    });
}

function printPreview() {
    window.print();
}

// -- Dictionary Page --
function loadDictionary() {
    fetch('/dictionary/items')
        .then(function(r) { return r.json(); })
        .then(function(d) { renderDictionary(d.items || {}); })
        .catch(function(e) { console.error('Failed to load dictionary:', e); });
}

function renderDictionary(items) {
    var container = document.getElementById('dictContent');
    if (!container) return;

    var zones = {
        'HOT LINE': { cls: 'hot', items: [] },
        'COLD LINE': { cls: 'cold', items: [] },
        'BAKERY/DESSERT': { cls: 'bakery', items: [] },
        'BEVERAGES': { cls: 'bev', items: [] },
    };

    var keys = Object.keys(items).sort();
    for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        var entry = items[key];
        var effectiveZone = entry.chef_override || entry.default_zone || 'HOT LINE';
        if (!zones[effectiveZone]) {
            zones[effectiveZone] = { cls: '', items: [] };
        }
        zones[effectiveZone].items.push({
            key: key,
            default_zone: entry.default_zone || '',
            chef_override: entry.chef_override || null,
            last_seen: entry.last_seen || '',
        });
    }

    var html = '';
    var zoneOrder = ['HOT LINE', 'COLD LINE', 'BAKERY/DESSERT', 'BEVERAGES'];
    for (var zi = 0; zi < zoneOrder.length; zi++) {
        var zone = zoneOrder[zi];
        var zd = zones[zone];
        if (!zd || !zd.items.length) continue;

        var headerBg = '';
        var headerColor = '';
        if (zone === 'HOT LINE') { headerBg = 'var(--hot-bg)'; headerColor = 'var(--hot)'; }
        else if (zone === 'COLD LINE') { headerBg = 'var(--cold-bg)'; headerColor = 'var(--cold)'; }
        else if (zone === 'BEVERAGES') { headerBg = 'var(--bev-bg)'; headerColor = 'var(--bev)'; }
        else if (zone === 'BAKERY/DESSERT') { headerBg = 'var(--bakery-bg)'; headerColor = 'var(--bakery)'; }

        html += '<div class="dict-zone-group">';
        html += '<div class="dict-zone-title" style="background:' + headerBg +
                ';color:' + headerColor + ';border-bottom:1px solid var(--border)">' +
                '<span class="zone-dot zone-' + zd.cls + '"></span>' +
                esc(zone) + ' (' + zd.items.length + ')</div>';

        html += '<table class="dict-table"><thead><tr>' +
                '<th>Item</th><th>Zone</th><th>Override</th><th>Actions</th>' +
                '</tr></thead><tbody>';

        for (var ii = 0; ii < zd.items.length; ii++) {
            var item = zd.items[ii];
            var displayName = item.key.replace(/\b\w/g, function(c) { return c.toUpperCase(); });

            html += '<tr data-key="' + esc(item.key) + '">';
            html += '<td>' + esc(displayName) + '</td>';
            html += '<td><select onchange="updateZone(\'' + esc(item.key).replace(/'/g, "\\'") +
                    '\', this.value)">' +
                    '<option value="HOT LINE"' + (zone === 'HOT LINE' ? ' selected' : '') + '>HOT LINE</option>' +
                    '<option value="COLD LINE"' + (zone === 'COLD LINE' ? ' selected' : '') + '>COLD LINE</option>' +
                    '<option value="BAKERY/DESSERT"' + (zone === 'BAKERY/DESSERT' ? ' selected' : '') + '>BAKERY/DESSERT</option>' +
                    '<option value="BEVERAGES"' + (zone === 'BEVERAGES' ? ' selected' : '') + '>BEVERAGES</option>' +
                    '</select></td>';

            if (item.chef_override) {
                html += '<td><span class="override-badge">OVERRIDE</span></td>';
                html += '<td><button class="dict-btn" onclick="resetOverride(\'' +
                        esc(item.key).replace(/'/g, "\\'") +
                        '\')">Reset</button></td>';
            } else {
                html += '<td><span style="color:var(--text-muted);font-size:11px">Default</span></td>';
                html += '<td></td>';
            }
            html += '</tr>';
        }
        html += '</tbody></table></div>';
    }

    if (!html) {
        html = '<div style="text-align:center;padding:48px;color:var(--text-muted)">' +
               '<h3 style="color:var(--text);margin-bottom:4px">Dictionary is empty</h3>' +
               '<p>Items will be added automatically when you process orders.</p></div>';
    }
    container.innerHTML = html;
}

function updateZone(itemKey, newZone) {
    fetch('/dictionary/update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ item: itemKey, zone: newZone, action: 'override' })
    })
    .then(function(r) { return r.json(); })
    .then(function(d) {
        if (d.success) {
            toast('Moved to ' + newZone);
            loadDictionary();
        } else {
            toast('Error: ' + (d.error || 'Unknown'));
        }
    })
    .catch(function(e) { toast('Error: ' + e.message); });
}

function resetOverride(itemKey) {
    fetch('/dictionary/update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ item: itemKey, action: 'reset' })
    })
    .then(function(r) { return r.json(); })
    .then(function(d) {
        if (d.success) {
            toast('Override removed');
            loadDictionary();
        }
    })
    .catch(function(e) { toast('Error: ' + e.message); });
}

function filterItems() {
    var query = document.getElementById('searchInput').value.toLowerCase();
    var rows = document.querySelectorAll('.dict-table tbody tr');
    for (var i = 0; i < rows.length; i++) {
        var key = (rows[i].getAttribute('data-key') || '').toLowerCase();
        rows[i].style.display = key.indexOf(query) >= 0 ? '' : 'none';
    }
}

function importDict() {
    var input = document.getElementById('importFile');
    if (input) input.click();
}

function handleImport(evt) {
    var file = evt.target.files[0];
    if (!file) return;
    var fd = new FormData();
    fd.append('file', file);
    fetch('/dictionary/import', { method: 'POST', body: fd })
        .then(function(r) { return r.json(); })
        .then(function(d) {
            if (d.success) {
                toast('Dictionary imported');
                loadDictionary();
            } else {
                toast('Error: ' + (d.error || 'Import failed'));
            }
        })
        .catch(function(e) { toast('Error: ' + e.message); });
    evt.target.value = '';
}
